﻿namespace MSI.VideoSamples.VideoPlayback.ViewModels
{
    using MSI.VideoSamples.VideoPlayback.Helpers;
    using MSI.VideoSamples.VideoPlayback.Models.Authentication;
    using MSI.VideoSamples.VideoPlayback.Models.Devices;
    using MSI.VideoSamples.VideoPlayback.Models.Events;
    using MSI.VideoSamples.VideoPlayback.Models.Ptz;
    using MSI.VideoSamples.VideoPlayback.Models.Snapshot;
    using MSI.VideoSamples.VideoPlayback.Models.System;
    using MSI.VideoSamples.VideoPlayback.Models.Video;
    using MSI.VMS;
    using System;
    using System.Collections.ObjectModel;
    using System.Linq;
    using System.Net.Http;
    using System.Reflection;
    using System.Threading;
    using System.Windows;
    using System.Windows.Controls;

    /// <summary>
    /// View model of the main application window.
    /// </summary>
    internal partial class MainWindowViewModel : BaseViewModel
    {
        private readonly ApplicationConfiguration applicationConfiguration;
        private readonly HttpClient httpClient;
        private readonly VmsSystemModel vmsSystemModel;
        private readonly AuthenticationModel authenticationModel;
        private readonly VideoModel videoModel;
        private readonly SnapshotModel snapshotModel;
        private readonly DevicesModel devicesModel;
        private readonly PtzModel ptzModel;
        private ObservableCollection<Device> devicesCollection;
        private ObservableCollection<Event> eventsCollection;
        private Event selectedEvent;
        private bool isCameraSelected;
        private bool isGeolocationAvailable;
        private TabItem selectedTab;
        private Timer deviceUpdatingTimer;

        /// <summary>
        /// Initializes instance of <see cref="MainWindowViewModel"/> class.
        /// </summary>
        public MainWindowViewModel()
        {
            this.deviceUpdatingTimer = new Timer((state) => this.SelectedDeviceUpdated(this.selectedDevice), null, Timeout.InfiniteTimeSpan, Timeout.InfiniteTimeSpan);

            this.eventsCollection = GlobalEventsCollection.CreateOrGetEventsCollection();

            this.VlcPath = $@"{AppDomain.CurrentDomain.BaseDirectory}\libvlc\win-x64";

            this.LoggingInStatus = new LoggingInStatus();
            httpClient = new HttpClient();

            this.applicationConfiguration = ApplicationConfiguration.GetFromRegistry();
            this.UserId = this.applicationConfiguration.UserId;
            this.Password = this.applicationConfiguration.Password;
            this.SubscriptionKey = this.applicationConfiguration.SubscriptionKey;
            this.ApiBaseUrl = this.applicationConfiguration.ApiBaseUrl;
            this.SnapshotWidth = 320;

            if (string.IsNullOrWhiteSpace(this.VlcPath))
            {
                return;
            }

            this.vmsSystemModel = new VmsSystemModel(this.httpClient);
            this.authenticationModel = new AuthenticationModel(this.httpClient);
            this.devicesModel = new DevicesModel(this.httpClient);
            this.ptzModel = new PtzModel(this.httpClient);
            this.snapshotModel = new SnapshotModel(this.httpClient);
            this.videoModel = new VideoModel(this.httpClient, this.VlcPath);
            this.VlcPlayer = this.videoModel.MediaPlayer;
            this.VlcPlayer.Playing += this.VlcPlayerPlaying;
            this.VlcPlayer.Stopped += this.VlcPlayerStopped;
            this.VlcPlayer.EndReached += this.VlcPlayerStopped;
            this.VlcPlayer.EncounteredError += this.VlcPlayerStopped;

            this.isSeekNearestEnabled = true;
            this.playbackState = PlaybackState.Stopped;

            OnPropertyChanged(nameof(this.VlcPlayer));
        }

        /// <summary>
        /// Gets string representing current application version.
        /// </summary>
        public string CurrentVersion
        {
            get
            {
                return Assembly.GetExecutingAssembly().GetName().Version.ToString(3);
            }
        }

        /// <summary>
        /// Gets or sets the selected tab.
        /// </summary>
        public TabItem SelectedTab
        {
            get
            {
                return this.selectedTab;
            }
            set
            {
                this.selectedTab = value;
                OnPropertyChanged("SelectedTab");
            }
        }

        /// <summary>
        /// Gets or sets selected device.
        /// </summary>
        public Device SelectedDevice
        {
            get
            {
                return this.selectedDevice;
            }
            set
            {
                SetValueOfSelectedDevice(value);
                OnPropertyChanged("SelectedDevice");
                OnPropertyChanged("IsGeolocationAvailable");
            }
        }

        private void SetValueOfSelectedDevice(Device value)
        {
            if (value != this.selectedDevice)
            {
                this.StopRtspStream();
            }

            this.selectedDevice = value;
            if (selectedDevice != null)
            {
                this.IsCameraSelected = false;
                this.deviceUpdatingTimer.Change(TimeSpan.FromSeconds(1), Timeout.InfiniteTimeSpan);
                this.isGeolocationAvailable = this.selectedDevice.Geolocation != null;
            }
            else
            {
                this.deviceUpdatingTimer.Change(Timeout.InfiniteTimeSpan, Timeout.InfiniteTimeSpan);
                this.SelectedDeviceDataSource = null;
                this.SelectedDeviceHasPtz = false;
                this.IsCameraSelected = false;
                this.isGeolocationAvailable = false;
            }
        }

        /// <summary>
        /// Gets or sets selected device data source.
        /// </summary>
        public DataSource SelectedDeviceDataSource
        {
            get
            {
                return this.selectedDeviceDataSource;
            }
            set
            {
                this.selectedDeviceDataSource = value;
                OnPropertyChanged("SelectedDeviceDataSource");
            }
        }

        /// <summary>
        /// Gets or sets collection of devices.
        /// </summary>
        public ObservableCollection<Device> DevicesCollection
        {
            get
            {
                return this.devicesCollection;
            }
            set
            {
                this.devicesCollection = value;
                OnPropertyChanged("DevicesCollection");
            }
        }

        /// <summary>
        /// Gets or sets selected event.
        /// </summary>
        public Event SelectedEvent
        {
            get
            {
                return this.selectedEvent;
            }
            set
            {
                this.selectedEvent = value;
                if (this.SelectedTab.Header.Equals("Events"))
                {
                    this.SelectedDevice = this.devicesCollection?.FirstOrDefault(camera => camera?.Id == this.selectedEvent?.Payload.CameraId);
                }
                OnPropertyChanged("SelectedEvent");
                OnPropertyChanged("IsEventCameraAvailable");
            }
        }

        /// <summary>
        /// Gets value indicating if selected event has associated camera
        /// </summary>
        public bool IsEventCameraAvailable
        {
            get
            {
                return (this.SelectedEvent != null) && this.IsCameraSelected;
            }
        }

        /// <summary>
        /// Gets or sets collection of events.
        /// </summary>
        public ObservableCollection<Event> EventsCollection
        {
            get
            {
                return this.eventsCollection;
            }
            set
            {
                this.eventsCollection = value;
                OnPropertyChanged("EventsCollection");
            }
        }

        /// <summary>
        /// Gets or sets value indicating if camera is selected.
        /// </summary>
        public bool IsCameraSelected
        {
            get
            {
                return this.isCameraSelected;
            }
            set
            {
                this.isCameraSelected = value;
                OnPropertyChanged("IsCameraSelected");
                OnPropertyChanged("IsEventCameraAvailable");
            }
        }

        /// <summary>
        /// Gets or sets value indicating if geolocation is available for selected camera.
        /// </summary>
        public bool IsGeolocationAvailable
        {
            get
            {
                return this.isGeolocationAvailable;
            }
            set
            {
                this.isGeolocationAvailable = value;
                OnPropertyChanged("IsGeolocationAvailable");
            }
        }

        /// <inheritdoc/>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (this.VlcPlayer != null)
                {
                    this.VlcPlayer.Playing -= VlcPlayerPlaying;
                    this.VlcPlayer.Stopped -= this.VlcPlayerStopped;
                    this.VlcPlayer.EndReached -= this.VlcPlayerStopped;
                    this.VlcPlayer.EncounteredError -= this.VlcPlayerStopped;
                }
                this.VlcPlayer?.Dispose();
                this.deviceUpdatingTimer?.Dispose();
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Updates all fields related to selected Device.
        /// </summary>
        /// <param name="device">
        /// The selected device.
        /// </param>
        private void SelectedDeviceUpdated(Device device)
        {
            if (device == null)
            {
                return;
            }

            Application.Current.Dispatcher.InvokeAsync(async () =>
            {
                DataSourceCollection deviceDataSources = await this.devicesModel.GetDataSourcesAsync(this.apiBaseUri, device);
                this.SelectedDeviceDataSource = deviceDataSources?.DataSources?.FirstOrDefault();
                this.SelectedDeviceHasPtz = this.SelectedDeviceDataSource?._Links?.PtzController != null;
                if (this.SelectedDeviceHasPtz)
                {
                    PtzController ptzController = await this.ptzModel.GetPtzControllerAsync(this.apiBaseUri, this.SelectedDeviceDataSource);
                    this.PtzPresetsCollection = await this.ptzModel.GetPtzPresetsCollectionAsync(this.apiBaseUri, ptzController).ConfigureAwait(false);
                    this.PtzToursCollection = await this.ptzModel.GetPtzToursCollectionAsync(this.apiBaseUri, ptzController).ConfigureAwait(false);
                    this.PtzHomeEnabled = ptzController._Links.RelHome != null;
                    this.PtzStopEnabled = ptzController._Links.RelHalt != null;
                }
                else
                {
                    this.PtzPresetsCollection = new ObservableCollection<PtzPreset>();
                    this.PtzToursCollection = new ObservableCollection<PtzTour>();
                    this.PtzHomeEnabled = false;
                    this.PtzStopEnabled = false;
                }
                this.IsCameraSelected = true;
            });
        }
    }
}
