﻿using LibVLCSharp.Shared;
using MSI.VideoSamples.VideoPlayback.Helpers;
using MSI.VMS;
using System;
using System.Globalization;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Windows.Input;

namespace MSI.VideoSamples.VideoPlayback.ViewModels
{
    /// <summary>
    /// View model of the main application window.
    /// </summary>
    internal partial class MainWindowViewModel : BaseViewModel
    {
        private string vlcPath;
        private bool isVideoPlaying;
        private bool isSeekNearestEnabled;
        private PlaybackKind playbackKind;
        private PlaybackState playbackState;
        private VideoQuality videoQuality = VideoQuality.low;
        private DateTime recordedVideoStartTime = DateTime.Now.AddMinutes(-1);
        private MediaPlayer mediaPlayer;

        /// <summary>
        /// Gets or sets VLC player.
        /// </summary>
        public MediaPlayer VlcPlayer
        {
            get
            {
                return this.mediaPlayer;
            }
            set
            {
                this.mediaPlayer = value;
                OnPropertyChanged("VlcPlayer");
            }
        }

        /// <summary>
        /// Gets or sets value indicating playback kind.
        /// </summary>
        public PlaybackKind PlaybackKind
        {
            get
            {
                return this.playbackKind;
            }
            set
            {
                this.playbackKind = value;
                OnPropertyChanged("PlaybackKind");
            }
        }

        /// <summary>
        /// Gets or sets value indicating playback state.
        /// </summary>
        public PlaybackState PlaybackState
        {
            get
            {
                return this.playbackState;
            }
            set
            {
                this.playbackState = value;
                OnPropertyChanged("PlaybackState");
                OnPropertyChanged("PtzControlsEnabled");
                OnPropertyChanged("PtzPresetsControlsEnabled");
                OnPropertyChanged("PtzPresetsTriggerButtonContent");
                OnPropertyChanged("PtzToursControlsEnabled");
                OnPropertyChanged("PtzToursStartButtonContent");
            }
        }

        /// <summary>
        /// Gets or sets recorded video start time.
        /// </summary>
        public DateTime RecordedVideoStartTime
        {
            get
            {
                return this.recordedVideoStartTime;
            }
            set
            {
                this.recordedVideoStartTime = value;
                OnPropertyChanged("RecordedVideoStartTime");
            }
        }

        /// <summary>
        /// Gets or sets value indicating if seeking the nearest available recording is enabled.
        /// </summary>
        public bool IsSeekNearestEnabled
        {
            get
            {
                return this.isSeekNearestEnabled;
            }
            set
            {
                this.isSeekNearestEnabled = value;
                OnPropertyChanged("IsSeekNearestEnabled");
            }
        }

        /// <summary>
        /// Gets or sets value indicating requested video quality.
        /// </summary>
        public VideoQuality VideoQuality
        {
            get
            {
                return this.videoQuality;
            }
            set
            {
                if (this.videoQuality != value)
                {
                    this.videoQuality = value;
                    this.StopRtspStream();
                }
                OnPropertyChanged("VideoQuality");
            }
        }

        /// <summary>
        /// Gets or sets value indicating that video is currently playing.
        /// </summary>
        public bool IsVideoPlaying
        {
            get
            {
                return this.isVideoPlaying;
            }
            set
            {
                this.isVideoPlaying = value;
                OnPropertyChanged("IsVideoPlaying");
            }
        }

        /// <summary>
        /// Gets or sets live video playback command.
        /// </summary>
        public ICommand LiveVideoCommand => new DelegateCommand(async (param) =>
        {
            await this.VideoCommand(this.VideoQuality, PlaybackKind.live);
        });

        /// <summary>
        /// Gets or sets recorded video playback command.
        /// </summary>
        public ICommand RecordedVideoCommand => new DelegateCommand(async (startTime) =>
        {
            if (startTime is DateTime)
            {
                await this.VideoCommand(this.VideoQuality, PlaybackKind.recorded, this.IsSeekNearestEnabled, (DateTime)startTime);
            }
        });

        /// <summary>
        /// Gets or sets alarm related video playback command.
        /// </summary>
        public ICommand RelatedVideoCommand => new DelegateCommand(async (param) =>
        {
            if (this.SelectedEvent == null)
            {
                return;
            }

            Match m = Regex.Match(this.SelectedEvent.Payload.EventStartTime, @"^(?<yr>\d{4})-?(?<mo>\d{2})-?(?<day>\d{2})T(?<hr>\d{2}):?(?<min>\d{2}):?(?<sec>\d{2})\.(?<msec>\d{1,3})Z?$");
            if (m.Success)
            {
                string result = new String('0', 23 - this.SelectedEvent.Payload.EventStartTime.Length);
                this.SelectedEvent.Payload.EventStartTime += result;
                string format = "yyyy-MM-ddTHH:mm:ss.fff";
                var dateTime = DateTime.ParseExact(
                    this.SelectedEvent.Payload.EventStartTime,
                    format,
                    CultureInfo.InvariantCulture,
                    DateTimeStyles.AssumeUniversal | DateTimeStyles.AdjustToUniversal);
                await this.VideoCommand(this.VideoQuality, PlaybackKind.recorded, this.IsSeekNearestEnabled, dateTime);
            }
        });

        /// <summary>
        /// Gets value indicating if VLC is installed.
        /// </summary>
        public bool IsVlcInstalled
        {
            get
            {
                return !string.IsNullOrWhiteSpace(this.VlcPath);
            }
        }

        /// <summary>
        /// Gets or sets path to VLC libraries.
        /// </summary>
        public string VlcPath
        {
            get
            {
                return this.vlcPath;
            }
            set
            {
                this.vlcPath = value;
                OnPropertyChanged("VlcPath");
                OnPropertyChanged("IsVlcInstalled");
            }
        }

        /// <summary>
        /// Handles VLC player Playing event.
        /// </summary>
        /// <param name="sender">The event sender.</param>
        /// <param name="e">The event arguments.</param>
        private void VlcPlayerPlaying(object sender, EventArgs e)
        {
            this.IsVideoPlaying = true;
            this.PlaybackState = this.PlaybackKind.Equals(PlaybackKind.live) ? PlaybackState.PlayingLive : PlaybackState.PlayingRecorded;
        }

        /// <summary>
        /// Handles VLC player stopped events.
        /// </summary>
        /// <param name="sender">The event sender.</param>
        /// <param name="e">The event arguments.</param>
        private void VlcPlayerStopped(object sender, EventArgs e)
        {
            this.PlaybackState = PlaybackState.Stopped;
        }

        /// <summary>
        /// Performs video playback initialization.
        /// </summary>
        /// <param name="videoQuality">Video quality (high or low).</param>
        /// <param name="playbackKind">Playback kind (live or recorded).</param>
        /// <param name="seekNearest">Indicates if the nearest recording should be played when exact time recording is not available.</param>
        /// <param name="videoStartTime">Video start time for recorded video.</param>
        /// <returns>Task performing video playback initialization.</returns>
        private async Task VideoCommand(VideoQuality videoQuality, PlaybackKind playbackKind, bool seekNearest = true, DateTime videoStartTime = default)
        {
            this.VlcPlayer.Pause();
            this.PlaybackState = PlaybackState.Initializing;
            string rtspUrl = await this.videoModel.PostRtspRequestAsync(this.apiBaseUri, this.SelectedDeviceDataSource, videoQuality, playbackKind, seekNearest, videoStartTime);
            await this.videoModel.PlayVideo(rtspUrl);
            this.PlaybackKind = playbackKind;
        }

        /// <summary>
        /// Stops RTSP stream playback.
        /// </summary>
        private void StopRtspStream()
        {
            this.PlaybackState = PlaybackState.Stopped;
            this.VlcPlayer.Stop();
        }
    }
}
