﻿namespace MSI.VideoSamples.VideoPlayback.ViewModels
{
    using MSI.VideoSamples.VideoPlayback.Helpers;
    using MSI.VMS;
    using Newtonsoft.Json;
    using System;
    using System.Globalization;
    using System.IO;
    using System.Linq;
    using System.Threading.Tasks;
    using System.Windows;
    using System.Windows.Forms;
    using System.Windows.Input;
    using static System.Environment;

    /// <summary>
    /// View model of the main application window.
    /// </summary>
    internal partial class MainWindowViewModel : BaseViewModel
    {
        private int snaphotWidth;

        /// <summary>
        /// Gets or sets the requested snapshot width.
        /// </summary>
        public int SnapshotWidth
        {
            get
            {
                return this.snaphotWidth;
            }
            set
            {
                this.snaphotWidth = value;
                OnPropertyChanged("SnapshotWidth");
            }
        }

        /// <summary>
        /// Gets or sets live snapshot command.
        /// </summary>
        public ICommand LiveSnapshotCommand => new DelegateCommand(async (param) =>
        {
            await this.SnapshotCommand(this.VideoQuality, this.SnapshotWidth);
        });

        /// <summary>
        /// Gets or sets recorded snapshot command.
        /// </summary>
        public ICommand RecordedSnapshotCommand => new DelegateCommand(async (snapshotTime) =>
        {
            if (snapshotTime is DateTime)
            {
                await this.SnapshotCommand(this.VideoQuality, this.SnapshotWidth, (DateTime)snapshotTime);
            }
        });

        /// <summary>
        /// Saves video snapshot to file.
        /// </summary>
        /// <param name="snapshotQuality">Snapshot quality (high or low).</param>
        /// <param name="snapshotWidth">Requested snapshot width in pixels.</param>
        /// <param name="snapshotTime">Snapshot time for recorded video snapshot. When <c>null</c> then live snapshot is taken.</param>
        /// <returns>Task performing video playback initialization.</returns>
        private async Task SnapshotCommand(VideoQuality snapshotQuality, int snapshotWidth = 320, DateTime? snapshotTime = null)
        {
            var multipartContent = await this.snapshotModel.PostSnapshotRequestAsync(this.apiBaseUri, this.SelectedDeviceDataSource, snapshotQuality, snapshotWidth, snapshotTime);

            var matadataContent = multipartContent.Contents.FirstOrDefault(content => content.Headers.ContentType.MediaType == "application/json");

            VMS.SnapshotMetadata snapshotMetadata = null;
            if (matadataContent != default)
            {
                using (Stream responseStream = await matadataContent.ReadAsStreamAsync())
                {
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            var responseString = streamReader.ReadToEnd();
                            snapshotMetadata = JsonConvert.DeserializeObject<VMS.SnapshotMetadata>(responseString);
                        }
                    }
                }
            }

            var jpegContent = multipartContent.Contents.FirstOrDefault(content => content.Headers.ContentType.MediaType == "image/jpeg");

            if (jpegContent != default)
            {
                using (Stream jpegStream = await jpegContent.ReadAsStreamAsync())
                {
                    if (jpegStream.Length > 0)
                    {
                        this.SaveJpegStreamToFile(jpegStream, this.GetSuggestedFileName(snapshotQuality, snapshotTime, snapshotMetadata));
                        return;
                    }
                }
            }

            System.Windows.MessageBox.Show("Failed to get the video snapshot", "Error", MessageBoxButton.OK, MessageBoxImage.Error);
        }

        /// <summary>
        /// Saves JPEG binary stream to file.
        /// </summary>
        /// <param name="jpegStream">JPEG binary stream.</param>
        /// <param name="fileName">The suggested filename.</param>
        private void SaveJpegStreamToFile(Stream jpegStream, string fileName)
        {
            using (SaveFileDialog fileSaveDialog = new SaveFileDialog()
            {
                Filter = "JPEG files|*.jpeg;*.jpg",
                FileName = fileName,
                DefaultExt = "jpg",
                AddExtension = true,
                OverwritePrompt = true,
                InitialDirectory = GetFolderPath(SpecialFolder.MyPictures),
                Title = "Save video snapshot",
                RestoreDirectory = true
            })
            {
                if (fileSaveDialog.ShowDialog() == DialogResult.OK)
                {
                    using (var fileStream = new FileStream(fileSaveDialog.FileName, FileMode.Create, FileAccess.Write))
                    {
                        jpegStream.CopyTo(fileStream);
                    }
                }
            }
        }

        /// <summary>
        /// Gets sanitized file name based on snapshot time and quality.
        /// </summary>
        /// <param name="snapshotTime">Snapshot time.</param>
        /// <param name="snapshotQuality">Snapshot quality.</param>
        /// <param name="snapshotMetadata">Snapshot meta data.</param>
        /// <returns>Sanitized file name.</returns>
        private string GetSuggestedFileName(VideoQuality snapshotQuality, DateTime? snapshotTime, SnapshotMetadata snapshotMetadata)
        {
            int? metadataWidth = snapshotMetadata?.MediaWidth;
            int? metadataHeight = snapshotMetadata?.MediaHeight;

            string snapshotSize = "";
            if ((metadataWidth != null) && (metadataHeight != null))
            {
                snapshotSize = $"_{metadataWidth}x{metadataHeight}";
            }

            string metadataTime = snapshotMetadata?.MediaTime;
            DateTime fileDate = snapshotTime.HasValue ? snapshotTime.Value.ToUniversalTime() : DateTime.Now.ToUniversalTime();
            if (metadataTime != null)
            {
                CultureInfo provider = CultureInfo.InvariantCulture;
                fileDate = DateTime.ParseExact(metadataTime, "yyyyMMddTHHmmss.fffZ", provider).ToUniversalTime();
            }

            var invalids = Path.GetInvalidFileNameChars().Union(new char[1] { ' ' }).ToArray();
            string fileNameTemplate = $"{selectedDevice.Name}_{fileDate.ToString("G")}_{snapshotQuality}{snapshotSize}.jpg";
            var fileName = string.Join("_", fileNameTemplate.Split(invalids, StringSplitOptions.RemoveEmptyEntries));
            return fileName;
        }
    }
}
