﻿namespace MSI.VideoSamples.VideoPlayback.ViewModels
{
    using MSI.VideoSamples.VideoPlayback.Helpers;
    using MSI.VMS;
    using System.Collections.ObjectModel;
    using System.Windows.Input;

    /// <summary>
    /// View model of the main application window.
    /// </summary>
    internal partial class MainWindowViewModel : BaseViewModel
    {
        private ObservableCollection<PtzPreset> ptzPresetsCollection;
        private ObservableCollection<PtzTour> ptzToursCollection;
        private Device selectedDevice;
        private DataSource selectedDeviceDataSource;
        private bool selectedDeviceHasPtz;
        private bool ptzHomeEnabled;
        private bool ptzStopEnabled;
        private PtzPreset selectedPtzPreset;
        private PtzTour selectedPtzTour;

        /// <summary>
        /// Gets or sets select PTZ preset.
        /// </summary>
        public PtzPreset SelectedPtzPreset
        {
            get
            {
                return this.selectedPtzPreset;
            }
            set
            {
                this.selectedPtzPreset = value;
                OnPropertyChanged("SelectedPtzPreset");
            }
        }

        /// <summary>
        /// Gets or sets select PTZ preset.
        /// </summary>
        public PtzTour SelectedPtzTour
        {
            get
            {
                return this.selectedPtzTour;
            }
            set
            {
                this.selectedPtzTour = value;
                OnPropertyChanged("SelectedPtzTour");
            }
        }

        /// <summary>
        /// Gets or sets value indicating selected device data source has PTZ Controller.
        /// </summary>
        public bool SelectedDeviceHasPtz
        {
            get
            {
                return this.selectedDeviceHasPtz;
            }
            set
            {
                this.selectedDeviceHasPtz = value;
                OnPropertyChanged("SelectedDeviceHasPtz");
                OnPropertyChanged("PtzControlsEnabled");
            }
        }

        /// <summary>
        /// Gets a value indicating if PTZ controls are enabled
        /// </summary>
        public bool PtzControlsEnabled
        {
            get
            {
                return this.selectedDeviceHasPtz && (this.playbackState == PlaybackState.PlayingLive);
            }
        }

        /// <summary>
        /// Gets a value indicating if PTZ Presets controls are enabled
        /// </summary>
        public bool PtzPresetsControlsEnabled
        {
            get
            {
                return this.selectedDeviceHasPtz && (this.ptzPresetsCollection?.Count > 0) && (this.playbackState == PlaybackState.PlayingLive);
            }
        }

        /// <summary>
        /// Gets a value indicating if PTZ Home control is enabled
        /// </summary>
        public bool PtzHomeEnabled
        {
            get
            {
                return this.ptzHomeEnabled;
            }
            set
            {
                this.ptzHomeEnabled = value;
                OnPropertyChanged("PtzHomeEnabled");
            }
        }

        /// <summary>
        /// Gets a value indicating if PTZ Stop control is enabled
        /// </summary>
        public bool PtzStopEnabled
        {
            get
            {
                return this.ptzStopEnabled;
            }
            set
            {
                this.ptzStopEnabled = value;
                OnPropertyChanged("PtzStopEnabled");
            }
        }

        /// <summary>
        /// Gets a value indicating if PTZ Tours controls are enabled
        /// </summary>
        public bool PtzToursControlsEnabled
        {
            get
            {
                return this.selectedDeviceHasPtz && (this.ptzToursCollection?.Count > 0) && (this.playbackState == PlaybackState.PlayingLive);
            }
        }

        /// <summary>
        /// Gets a value of PTZ Trigger preset button content
        /// </summary>
        public string PtzPresetsTriggerButtonContent
        {
            get
            {
                return PtzPresetsControlsEnabled ? "Go to Preset" : "No presets";
            }
        }

        /// <summary>
        /// Gets a value of PTZ Start tour button content
        /// </summary>
        public string PtzToursStartButtonContent
        {
            get
            {
                return PtzToursControlsEnabled ? "Start Tour" : "No tours";
            }
        }

        /// <summary>
        /// Gets or sets collection of PTZ Presets.
        /// </summary>
        public ObservableCollection<PtzPreset> PtzPresetsCollection
        {
            get
            {
                return this.ptzPresetsCollection;
            }
            set
            {
                this.ptzPresetsCollection = value;
                OnPropertyChanged("PtzPresetsCollection");
                OnPropertyChanged("PtzPresetsControlsEnabled");
                OnPropertyChanged("PtzPresetsTriggerButtonContent");
            }
        }

        /// <summary>
        /// Gets or sets collection of PTZ Tours.
        /// </summary>
        public ObservableCollection<PtzTour> PtzToursCollection
        {
            get
            {
                return this.ptzToursCollection;
            }
            set
            {
                this.ptzToursCollection = value;
                OnPropertyChanged("PtzToursCollection");
                OnPropertyChanged("PtzToursControlsEnabled");
                OnPropertyChanged("PtzToursStartButtonContent");
            }
        }

        /// <summary>
        /// Moves PTZ camera to its Home Position.
        /// </summary>
        public ICommand PtzGoToHomeCommand => new DelegateCommand(async (param) =>
        {
            PtzController ptzController = await this.ptzModel.GetPtzControllerAsync(this.apiBaseUri, this.SelectedDeviceDataSource);
            await this.ptzModel.GoToHomePositionAsync(this.apiBaseUri, ptzController);
        });

        /// <summary>
        /// Triggers currently selected PTZ Preset on a PTZ Camera.
        /// </summary>
        public ICommand PtzTriggerPresetCommand => new DelegateCommand(async (param) =>
        {
            if (this.SelectedPtzPreset == null)
            {
                return;
            }

            await this.ptzModel.TriggerPtzPresetAsync(this.apiBaseUri, this.SelectedPtzPreset);
        });

        /// <summary>
        /// Stops PTZ tour or pattern.
        /// </summary>
        public ICommand PtzStopCommand => new DelegateCommand(async (param) =>
        {
            PtzController ptzController = await this.ptzModel.GetPtzControllerAsync(this.apiBaseUri, this.SelectedDeviceDataSource);
            await this.ptzModel.StopAsync(this.apiBaseUri, ptzController);
        });

        /// <summary>
        /// Triggers currently selected PTZ Preset on a PTZ Camera.
        /// </summary>
        public ICommand PtzStartTourCommand => new DelegateCommand(async (param) =>
        {
            if (this.SelectedPtzTour == null)
            {
                return;
            }

            await this.ptzModel.StartPtzTourAsync(this.apiBaseUri, this.SelectedPtzTour);
        });
    }
}
