﻿namespace MSI.VideoSamples.VideoPlayback.ViewModels
{
    using MSI.VideoSamples.VideoPlayback.Helpers;
    using MSI.VMS;
    using System;
    using System.Windows.Forms;
    using System.Windows.Input;

    /// <summary>
    /// View model of the main application window.
    /// </summary>
    internal partial class MainWindowViewModel : BaseViewModel
    {
        private const string ApiKey = "ApiKey";
        private string userId;
        private string password;
        private string subscriptionKey;
        private string apiBaseUrl;
        private Uri apiBaseUri;
        private bool isUserLoggedIn;
        private bool isLoggingInInProgress;
        private UserInformation userInformation;
        private LoggingInStatus loggingInStatus;

        /// <summary>
        /// Gets or sets base URL of VMS API.
        /// </summary>
        public string ApiBaseUrl
        {
            get
            {
                return this.apiBaseUrl;
            }
            set
            {
                this.apiBaseUrl = value;
                this.IsUserLoggedIn = false;
                this.applicationConfiguration.ApiBaseUrl = this.ApiBaseUrl;
                this.apiBaseUri = !string.IsNullOrWhiteSpace(this.ApiBaseUrl) ? new Uri(this.ApiBaseUrl) : default;
                OnPropertyChanged("ApiBaseUrl");
            }
        }

        /// <summary>
        /// Gets or sets API key.
        /// </summary>
        public string SubscriptionKey
        {
            get
            {
                return subscriptionKey;
            }
            set
            {
                this.subscriptionKey = value;
                this.IsUserLoggedIn = false;
                this.applicationConfiguration.SubscriptionKey = this.SubscriptionKey;
                this.httpClient?.DefaultRequestHeaders.Remove(ApiKey);
                this.httpClient?.DefaultRequestHeaders.Add(ApiKey, this.SubscriptionKey);
                OnPropertyChanged("SubscriptionKey");
            }
        }

        /// <summary>
        /// Gets or set user identifier.
        /// </summary>
        public string UserId
        {
            get
            {
                return this.userId;
            }
            set
            {
                this.userId = value;
                this.IsUserLoggedIn = false;
                this.applicationConfiguration.UserId = this.UserId;
                OnPropertyChanged("UserId");
            }
        }

        /// <summary>
        /// Gets or sets user password.
        /// </summary>
        public string Password
        {
            get
            {
                return this.password;
            }
            set
            {
                this.password = value;
                this.IsUserLoggedIn = false;
                this.applicationConfiguration.Password = this.password;
                OnPropertyChanged("Password");
            }
        }

        /// <summary>
        /// Gets or sets user information.
        /// </summary>
        public UserInformation UserInformation
        {
            get
            {
                return this.userInformation;
            }
            set
            {
                this.userInformation = value;
                OnPropertyChanged("UserInformation");
            }
        }

        /// <summary>
        /// Gets or sets value indicating if user is logged in.
        /// </summary>
        public bool IsUserLoggedIn
        {
            get
            {
                return this.isUserLoggedIn;
            }
            set
            {
                this.isUserLoggedIn = value;
                this.UserInformation = this.userInformation;
                if (!this.isUserLoggedIn)
                {
                    this.DevicesCollection = default;
                    this.UserInformation = default;
                }
                this.LoggingInStatus.IsUserLoggedIn = value;
                OnPropertyChanged("LoggingInStatus");
                OnPropertyChanged("IsUserLoggedIn");
            }
        }

        /// <summary>
        /// Gets or sets value indicating if logging in is in progress.
        /// </summary>
        public bool IsLoggingInInProgress
        {
            get
            {
                return this.isLoggingInInProgress;
            }
            set
            {
                this.isLoggingInInProgress = value;
                this.LoggingInStatus.IsLoggingInInProgress = value;
                OnPropertyChanged("LoggingInStatus");
                OnPropertyChanged("IsLoggingInInProgress");
            }
        }

        /// <summary>
        /// Gets or sets value describing information about logging in process.
        /// </summary>
        public LoggingInStatus LoggingInStatus
        {
            get
            {
                return this.loggingInStatus;
            }
            set
            {
                this.loggingInStatus = value;
                OnPropertyChanged("LoggingInStatus");
            }
        }

        /// <summary>
        /// Gets or sets login command.
        /// </summary>
        public ICommand LoginCommand => new DelegateCommand(async (param) =>
        {
            StopRtspStream();
            SetValueOfLoggingInStatusVariables(false, true);

            VmsSystem systemInformation = await this.vmsSystemModel.GetSystemInforamtionAsync(this.apiBaseUri);
            UserCredentials userCredentials = new VMS.UserCredentials() { Email = this.UserId, Password = this.Password };

            try
            {
                this.userInformation = await this.authenticationModel.PostLoginAsync(this.apiBaseUri, userCredentials, systemInformation);
            }
            catch (Exception ex)
            {
                SetValueOfLoggingInStatusVariables(false, false);
                MessageBox.Show(ex.Message, "Logging in process status");
                return;
            }

            systemInformation = await this.vmsSystemModel.GetSystemInforamtionAsync(this.apiBaseUri);
            this.DevicesCollection = await this.devicesModel.GetDevicesAsync(this.apiBaseUri, systemInformation);
            this.applicationConfiguration.SaveToRegistry();

            SetValueOfLoggingInStatusVariables(true, false);
        });

        private void SetValueOfLoggingInStatusVariables(bool isUserLoggedIn, bool isLoggingInInProgress)
        {
            this.IsUserLoggedIn = isUserLoggedIn;
            this.IsLoggingInInProgress = isLoggingInInProgress;
        }
    }
}
