﻿using System;
using System.IO;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;

namespace MSI.VideoSamples.VideoPlayback.Models.Video
{
    using LibVLCSharp.Shared;
    using MSI.VideoSamples.VideoPlayback.Extensions;
    using MSI.VMS;
    using Newtonsoft.Json;

    /// <summary>
    /// Represents Video API model.
    /// </summary>
    internal class VideoModel : IDisposable
    {
        private const string contentTypeJson = "application/json";

        /// <summary>
        /// Holds a read-only reference to array of VLC player initialization parameters.
        /// </summary>
        private readonly string[] vlcOptions =  {
            "--rtsp-tcp",
            "--rtsp-http",
            "--rtsp-http-port=5544",
            "--no-sap-parse",
            "--no-lua"
        };

        /// <summary>
        /// Holds a read-only reference to VLC media player.
        /// </summary>
        private readonly MediaPlayer mediaPlayer;

        /// <summary>
        /// Holds a read-only reference to <see cref="HttpClient" instance./>
        /// </summary>
        private readonly HttpClient httpClient;

        /// <summary>
        /// Holds a read-only reference to <see cref="LibVLC" instance.>
        /// </summary>
        private readonly LibVLC libVlc;

        /// <summary>
        /// Indicates if this instance is disposed.
        /// </summary>
        private bool disposedValue;

        /// <summary>
        /// Initializes instance of <see cref="VideoModel"/> class.
        /// </summary>
        /// <param name="httpClient">
        /// <see cref="HttpClient"./> instance.
        /// </param>
        internal VideoModel(HttpClient httpClient, string vlcPath)
        {
            this.httpClient = httpClient;

            Core.Initialize(vlcPath);
            this.libVlc = new LibVLC(this.vlcOptions);
            this.mediaPlayer = new MediaPlayer(this.libVlc);
        }

        /// <summary>
        /// Gets instance of VLC media player.
        /// </summary>
        internal MediaPlayer MediaPlayer => this.mediaPlayer;

        /// <summary>
        /// Asynchronously posts request to initialize RTSP stream from specified data source.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="dataSource">Instance of <see cref="DataSource"/>.</param>
        /// <param name="videoQuality">Video quality (high or low).</param>
        /// <param name="playbackKind">Playback kind (live or recorded).</param>
        /// <param name="seekNearest">Indicates if the nearest recording should be played when exact time recording is not available.</param>
        /// <param name="videoStartTime">Video start time for recorded video.</param>
        /// <returns>Task returning URL of RTSP stream.</returns>
        internal async Task<string> PostRtspRequestAsync(Uri baseApiUri, DataSource dataSource, VideoQuality videoQuality, PlaybackKind playbackKind, bool seekNearest, DateTime videoStartTime)
        {
            if (dataSource == null)
            {
                throw new ArgumentNullException(nameof(dataSource));
            }

            if (dataSource._Links == null)
            {
                throw new ArgumentException(nameof(dataSource._Links));
            }

            string rtspInitializeUrl = dataSource._Links?.Rtsp;

            string cameraId = dataSource.Id;

            string json = await Task.Run(() => JsonConvert.SerializeObject(new VMS.RtspRequest()
            {
                CameraId = cameraId,
                Media = "video",
                Quality = videoQuality.ToString(),
                Search = seekNearest,
                T = playbackKind.Equals(PlaybackKind.live) ? "live" : videoStartTime.ToUniversalTime().ToString("yyyyMMddTHHmmss.fffZ")
            }));

            Uri requestUri = new Uri(baseApiUri, rtspInitializeUrl);

            using (HttpContent httpContent = new StringContent(json, Encoding.UTF8, contentTypeJson))
            using (HttpResponseMessage httpGetSystemInformationResponse = await this.httpClient.PostAsync(requestUri, httpContent))
            {
                httpGetSystemInformationResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
                using (Stream responseStream = await httpGetSystemInformationResponse?.Content.ReadAsStreamAsync())
                {
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            var responseString = streamReader.ReadToEnd();
                            return JsonConvert.DeserializeObject<VMS.RtspResponse>(responseString).RtspRelayURL;
                        }
                    }
                    else
                    {
                        return null;
                    }
                }
            }
        }

        /// <summary>
        /// Asynchronously initializes live video playback from RTSP URL.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="videoUrl">URL of RTSP video stream.</param>
        /// <returns>Task initializing live video playback.</returns>
        internal async Task PlayVideo(string videoUrl)
        {
            var media = new Media(this.libVlc, videoUrl, FromType.FromLocation);
            await media.Parse(MediaParseOptions.FetchNetwork);
            this.mediaPlayer.Media = media;
            this.mediaPlayer.Play();
        }

        /// <inheritdoc/>
        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    this.mediaPlayer?.Dispose();
                    this.libVlc?.Dispose();
                }

                disposedValue = true;
            }
        }

        /// <inheritdoc/>
        public void Dispose()
        {
            // Do not change this code. Put cleanup code in 'Dispose(bool disposing)' method
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }
    }
}
