﻿namespace MSI.VideoSamples.VideoPlayback.Models.Snapshot
{
    using global::System;
    using global::System.Net.Http;
    using global::System.Text;
    using global::System.Threading.Tasks;
    using MSI.VideoSamples.VideoPlayback.Extensions;
    using MSI.VMS;
    using Newtonsoft.Json;

    /// <summary>
    /// Represents Snapshots API model.
    /// </summary>
    internal class SnapshotModel : IDisposable
    {
        private const string contentTypeJson = "application/json";

        /// <summary>
        /// Holds a read-only reference to <see cref="HttpClient" instance./>
        /// </summary>
        private readonly HttpClient httpClient;

        /// <summary>
        /// Indicates if this instance is disposed.
        /// </summary>
        private bool disposedValue;

        /// <summary>
        /// Initializes instance of <see cref="VideoModel"/> class.
        /// </summary>
        /// <param name="httpClient">
        /// <see cref="HttpClient"./> instance.
        /// </param>
        internal SnapshotModel(HttpClient httpClient)
        {
            this.httpClient = httpClient;
        }

        /// <summary>
        /// Asynchronously posts request to get snapshot from specified data source.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="dataSource">Instance of <see cref="DataSource"/>.</param>
        /// <param name="snapshotQuality">Snapshot quality (high or low).</param>
        /// <param name="snapshotWidth">Requested snapshot width in pixels.</param>
        /// <param name="snapshotTime">Time of the snapshot (<c>live</c> for live video snapshot).</param>
        /// <returns>Task returning snapshot file stream.</returns>
        internal async Task<MultipartMemoryStreamProvider> PostSnapshotRequestAsync(Uri baseApiUri, DataSource dataSource, VideoQuality snapshotQuality, int snapshotWidth, DateTime? snapshotTime)
        {
            if (dataSource == null)
            {
                throw new ArgumentNullException(nameof(dataSource));
            }

            if (dataSource._Links == null)
            {
                throw new ArgumentException(nameof(dataSource._Links));
            }

            string snapshotUrl = dataSource._Links?.Snapshot;

            string json = await Task.Run(() => JsonConvert.SerializeObject(new VMS.SnapshotRequest()
            {
                Quality = snapshotQuality.ToString(),
                Time = snapshotTime?.ToUniversalTime().ToString("yyyyMMddTHHmmss.fffZ"),
                Width = snapshotWidth
            }, new JsonSerializerSettings { NullValueHandling = NullValueHandling.Ignore }));

            var requestUri = new Uri(baseApiUri, snapshotUrl);

            using (HttpContent httpContent = new StringContent(json, Encoding.UTF8, contentTypeJson))
            using (HttpResponseMessage httpPostSnapshotResponse = await this.httpClient.PostAsync(requestUri, httpContent))
            {
                httpPostSnapshotResponse.EnsureSuccessStatusCodeAndGetInnerMessage();

                return await httpPostSnapshotResponse?.Content.ReadAsMultipartAsync();
            }
        }

        /// <inheritdoc/>
        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    //
                }

                disposedValue = true;
            }
        }

        /// <inheritdoc/>
        public void Dispose()
        {
            // Do not change this code. Put cleanup code in 'Dispose(bool disposing)' method
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }
    }
}
