﻿using System;
using System.IO;
using System.Net.Http;
using System.Threading.Tasks;
using System.Collections.ObjectModel;

namespace MSI.VideoSamples.VideoPlayback.Models.Ptz
{
    using MSI.VideoSamples.VideoPlayback.Extensions;
    using MSI.VMS;
    using Newtonsoft.Json;

    /// <summary>
    /// Represents PTZ API model.
    /// </summary>
    internal class PtzModel : IDisposable
    {
        /// <summary>
        /// Holds a read-only reference to <see cref="HttpClient" instance./>
        /// </summary>
        private readonly HttpClient httpClient;

        /// <summary>
        /// Indicates if this instance is disposed.
        /// </summary>
        private bool disposedValue;

        /// <summary>
        /// Initializes instance of <see cref="PtzModel"/> class.
        /// </summary>
        /// <param name="httpClient">
        /// <see cref="HttpClient"./> instance.
        /// </param>
        internal PtzModel(HttpClient httpClient)
        {
            this.httpClient = httpClient;
        }

        /// <summary>
        /// Asynchronously posts request to get PTZ Controller from specified data source.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="dataSource">Instance of <see cref="DataSource"/>.</param>
        /// <returns>Task returning URL of RTSP stream.</returns>
        internal async Task<PtzController> GetPtzControllerAsync(Uri baseApiUri, DataSource dataSource)
        {
            if (dataSource == null)
            {
                throw new ArgumentNullException(nameof(dataSource));
            }

            if (dataSource._Links == null)
            {
                throw new ArgumentException(nameof(dataSource._Links));
            }

            if (dataSource._Links.PtzController == null)
            {
                return null;
            }

            string ptzControllerUrl = dataSource._Links?.PtzController;

            Uri requestUri = new Uri(baseApiUri, ptzControllerUrl);

            using (HttpResponseMessage httpGetPtzControllerResponse = await this.httpClient.GetAsync(requestUri))
            {
                httpGetPtzControllerResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
                using (Stream responseStream = await httpGetPtzControllerResponse?.Content.ReadAsStreamAsync())
                {
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            var responseString = streamReader.ReadToEnd();
                            return JsonConvert.DeserializeObject<VMS.PtzController>(responseString);
                        }
                    }
                    else
                    {
                        return null;
                    }
                }
            }
        }

        /// <summary>
        /// Asynchronously requests to move camera to its Home Position.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="selectedDevice">Instance of <see cref="Device"/>.</param>
        /// <returns>Task returning <see cref="DataSourceCollection"/>.</returns>
        internal async Task GoToHomePositionAsync(Uri baseApiUri, PtzController ptzController)
        {
            if (ptzController?._Links?.RelHome == null)
            {
                return;
            }

            Uri requestUri = new Uri(baseApiUri, ptzController._Links.RelHome);

            using (HttpResponseMessage httpGetSystemInformationResponse = await this.httpClient.PostAsync(requestUri, null))
            {
                httpGetSystemInformationResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
            }
        }

        /// <summary>
        /// Asynchronously requests to stop PTZ tour or pattern.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="selectedDevice">Instance of <see cref="Device"/>.</param>
        /// <returns>Task returning <see cref="DataSourceCollection"/>.</returns>
        internal async Task StopAsync(Uri baseApiUri, PtzController ptzController)
        {
            if (ptzController?._Links?.RelHalt == null)
            {
                return;
            }

            Uri requestUri = new Uri(baseApiUri, ptzController._Links.RelHalt);

            using (HttpResponseMessage httpGetSystemInformationResponse = await this.httpClient.PostAsync(requestUri, null))
            {
                httpGetSystemInformationResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
            }
        }

        /// <summary>
        /// Asynchronously requests to get collection of PTZ Presets.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="ptzController">Instance of <see cref="PtzController"/>.</param>
        /// <returns>Task returning observable collection of <see cref="PtzPreset">.</returns>
        internal async Task<ObservableCollection<PtzPreset>> GetPtzPresetsCollectionAsync(Uri baseApiUri, PtzController ptzController)
        {
            if (ptzController?._Links?.RelPresets == null)
            {
                return new ObservableCollection<PtzPreset>();
            }

            var ptzPresetsObservableCollection = new ObservableCollection<PtzPreset>();

            Uri requestUri = new Uri(baseApiUri, ptzController._Links.RelPresets);

            using (HttpResponseMessage httpGetPtzPresetsCollectionResponse = await this.httpClient.GetAsync(requestUri))
            {
                httpGetPtzPresetsCollectionResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
                using (Stream responseStream = await httpGetPtzPresetsCollectionResponse?.Content.ReadAsStreamAsync())
                {
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            var responseString = streamReader.ReadToEnd();
                            var ptzPresets = JsonConvert.DeserializeObject<VMS.PtzPresetsCollection>(responseString).PtzPresets;
                            foreach (PtzPreset preset in ptzPresets)
                            {
                                ptzPresetsObservableCollection.Add(preset);
                            }
                            return ptzPresetsObservableCollection;
                        }
                    }
                    else
                    {
                        return new ObservableCollection<PtzPreset>();
                    }
                }
            }
        }

        /// <summary>
        /// Asynchronously triggers selected PTZ Preset.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="selectedPtzPreset">Instance of <see cref="PtzPreset"/>.</param>
        internal async Task TriggerPtzPresetAsync(Uri baseApiUri, PtzPreset selectedPtzPreset)
        {
            if (selectedPtzPreset?._Links?.RelTrigger == null)
            {
                return;
            }

            Uri requestUri = new Uri(baseApiUri, selectedPtzPreset._Links.RelTrigger);

            using (HttpResponseMessage httpGetSystemInformationResponse = await this.httpClient.PostAsync(requestUri, null))
            {
                httpGetSystemInformationResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
            }
        }

        /// <summary>
        /// Asynchronously requests to get collection of PTZ Tours.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="ptzController">Instance of <see cref="PtzController"/>.</param>
        /// <returns>Task returning observable collection of <see cref="PtzTour">.</returns>
        internal async Task<ObservableCollection<PtzTour>> GetPtzToursCollectionAsync(Uri baseApiUri, PtzController ptzController)
        {
            if (ptzController?._Links?.RelTours == null)
            {
                return new ObservableCollection<PtzTour>();
            }

            var ptzToursObservableCollection = new ObservableCollection<PtzTour>();

            Uri requestUri = new Uri(baseApiUri, ptzController._Links.RelTours);

            using (HttpResponseMessage httpGetPtzToursCollectionResponse = await this.httpClient.GetAsync(requestUri))
            {
                httpGetPtzToursCollectionResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
                using (Stream responseStream = await httpGetPtzToursCollectionResponse?.Content.ReadAsStreamAsync())
                {
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            var responseString = streamReader.ReadToEnd();
                            var ptzTours = JsonConvert.DeserializeObject<VMS.PtzToursCollection>(responseString).PtzTours;
                            foreach (PtzTour tour in ptzTours)
                            {
                                ptzToursObservableCollection.Add(tour);
                            }
                            return ptzToursObservableCollection;
                        }
                    }
                    else
                    {
                        return new ObservableCollection<PtzTour>();
                    }
                }
            }
        }

        /// <summary>
        /// Asynchronously starts selected PTZ Tour.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="selectedPtzTour">Instance of <see cref="PtzTour"/>.</param>
        internal async Task StartPtzTourAsync(Uri baseApiUri, PtzTour selectedPtzTour)
        {
            if (selectedPtzTour?._Links?.RelStart == null)
            {
                return;
            }

            Uri requestUri = new Uri(baseApiUri, selectedPtzTour._Links.RelStart);

            using (HttpResponseMessage httpGetSystemInformationResponse = await this.httpClient.PostAsync(requestUri, null))
            {
                httpGetSystemInformationResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
            }
        }

        /// <inheritdoc/>
        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    //
                }

                disposedValue = true;
            }
        }

        /// <inheritdoc/>
        public void Dispose()
        {
            // Do not change this code. Put cleanup code in 'Dispose(bool disposing)' method
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }
    }
}
