﻿using System;
using System.Collections.ObjectModel;
using System.IO;
using System.Net.Http;
using System.Threading.Tasks;

namespace MSI.VideoSamples.VideoPlayback.Models.Devices
{
    using MSI.VideoSamples.VideoPlayback.Extensions;
    using MSI.VMS;
    using Newtonsoft.Json;

    /// <summary>
    /// Represents Devices API model.
    /// </summary>
    internal class DevicesModel : IDisposable
    {
        /// <summary>
        /// Holds a read-only reference to <see cref="HttpClient" instance./>
        /// </summary>
        private readonly HttpClient httpClient;

        /// <summary>
        /// Indicates if this instance is disposed.
        /// </summary>
        private bool disposedValue;

        /// <summary>
        /// Initializes instance of <see cref="DevicesModel"/> class.
        /// </summary>
        /// <param name="httpClient">
        /// <see cref="HttpClient"./> instance.
        /// </param>
        internal DevicesModel(HttpClient httpClient)
        {
            this.httpClient = httpClient;
        }

        /// <summary>
        /// Asynchronously gets list of devices in the system.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="systemInformation">Instance of <see cref="VmsSystem"/>.</param>
        /// <returns>Task returning observable collection of <see cref="Device">.</returns>
        internal async Task<ObservableCollection<Device>> GetDevicesAsync(Uri baseApiUri, VmsSystem systemInformation)
        {
            var devicesCollection = new ObservableCollection<Device>();
            var link = systemInformation._Links.RelDevices;

            do
            {
                Uri requestUri = new Uri(baseApiUri, link);

                using (HttpResponseMessage httpGetPageResponse = await this.httpClient.GetAsync(requestUri))
                {
                    httpGetPageResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
                    using (Stream pageResponseStream = await httpGetPageResponse?.Content.ReadAsStreamAsync())
                    {
                        if (pageResponseStream != null)
                        {
                            using (var pageStreamReader = new StreamReader(pageResponseStream))
                            {
                                var pageResponseString = pageStreamReader.ReadToEnd();
                                AddDeviesToCollection(devicesCollection, pageResponseString);
                                if (!(JsonConvert.DeserializeObject<VMS.DevicesCollection>(pageResponseString).CollectionHeader.Links is null))
                                {
                                    link = JsonConvert.DeserializeObject<VMS.DevicesCollection>(pageResponseString).CollectionHeader.Links.RelNext;
                                }
                                else
                                {
                                    link = String.Empty;
                                }
                            }
                        }
                    }
                }
            } while (!String.IsNullOrEmpty(link));

            return devicesCollection;
        }

        /// <summary>
        /// Asynchronously gets list of data sources for given device.
        /// </summary>
        /// <param name="baseApiUri">Base API URI.</param>
        /// <param name="selectedDevice">Instance of <see cref="Device"/>.</param>
        /// <returns>Task returning <see cref="DataSourceCollection"/>.</returns>
        internal async Task<VMS.DataSourceCollection> GetDataSourcesAsync(Uri baseApiUri, Device selectedDevice)
        {
            Uri requestUri = new Uri(baseApiUri, selectedDevice._Links.RelDataSources);

            using (HttpResponseMessage httpGetSystemInformationResponse = await this.httpClient.GetAsync(requestUri))
            {
                httpGetSystemInformationResponse.EnsureSuccessStatusCodeAndGetInnerMessage();
                using (Stream responseStream = await httpGetSystemInformationResponse?.Content.ReadAsStreamAsync())
                {
                    if (responseStream != null)
                    {
                        using (var streamReader = new StreamReader(responseStream))
                        {
                            var responseString = streamReader.ReadToEnd();
                            return JsonConvert.DeserializeObject<VMS.DataSourceCollection>(responseString);
                        }
                    }
                    else
                    {
                        return null;
                    }
                }
            }
        }

        /// <inheritdoc/>
        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    //
                }

                disposedValue = true;
            }
        }

        /// <inheritdoc/>
        public void Dispose()
        {
            // Do not change this code. Put cleanup code in 'Dispose(bool disposing)' method
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        private void AddDeviesToCollection(ObservableCollection<Device> devicesCollection, string responseString)
        {
            var devicesOnPage = JsonConvert.DeserializeObject<VMS.DevicesCollection>(responseString).Devices;
            foreach (Device device in devicesOnPage)
            {
                devicesCollection.Add(device);
            }
        }
    }
}
