﻿using System;
using System.Linq;
using System.Net.Http;
using System.Security.Cryptography;

namespace MSI.VideoSamples.VideoPlayback.Helpers
{
    public static class MessageRequestValidator
    {
        /// <summary>
        /// Computes message signatures (hash value).
        /// </summary>
        /// <param name="messagePayload">
        /// The message payload (JSON).
        /// </param>
        /// <param name="authenticationToken">
        /// Authentication token - must be equal to value set in the ACS webhook definition.
        /// </param>
        /// <returns>
        /// Message signature.
        /// </returns>
        public static string ComputeMessageSignature(string messagePayload, string authenticationToken)
        {
            var content = System.Text.Encoding.UTF8.GetBytes(messagePayload);
            var token = System.Text.Encoding.UTF8.GetBytes(authenticationToken);
            using (HMACSHA256 hmacSha256 = new HMACSHA256(token))
            {
                return Convert.ToBase64String(hmacSha256.ComputeHash(content));
            }
        }

        /// <summary>
        /// Validates if message signature is correct.
        /// </summary>
        /// <param name="request">
        /// HTTP request received (ACS alarm sent through webhook).
        /// </param>
        /// <param name="authenticationToken">
        /// Authentication token - must be equal to value set in the ACS webhook definition.
        /// </param>
        /// <returns>
        /// <see langword="true"/> if message is valid, <see langword="false"/> otherwise.
        /// </returns>
        public static bool ValidateMessageRequest(HttpRequestMessage request, string authenticationToken)
        {
            var authHeader = request.Headers.GetValues("Authorization").First();
            if (authHeader != null && authHeader.StartsWith("HMAC-SHA256"))
            {
                var headerHmac = authHeader.Substring("HMAC-SHA256 ".Length).Trim();
                var rawMessage = request.Content.ReadAsStringAsync().Result;
                var computedHmac = ComputeMessageSignature(rawMessage, authenticationToken);
                return headerHmac.Equals(computedHmac);
            }
            return false;
        }
    }
}
